/*
 * Copyright (C) 2023 Chris Talbot
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#include "config.h"
#include "geoclue-stumbler-submission-marker.h"

#define _GNU_SOURCE
#include <glib.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <math.h>
#include <graphene.h>

struct _GeoclueStumblerSubmissionMarker
{
  ShumateMarker        parent_instance;

  GtkPopover          *popover;
  GtkImage            *marker_image;
  GtkLabel            *marker_time_label;
  GtkLabel            *marker_cell_tower_label;
  GtkLabel            *marker_wifi_ap_label;
};

G_DEFINE_TYPE (GeoclueStumblerSubmissionMarker, geoclue_stumbler_submission_marker, SHUMATE_TYPE_MARKER)

static void
marker_clicked_cb (GtkGestureLongPress *gesture,
                   int                  n_press,
                   gdouble              x,
                   gdouble              y,
                   GeoclueStumblerSubmissionMarker    *self)
{
  if (n_press != 1)
    return;

  if (!gtk_widget_get_parent (GTK_WIDGET (self->popover)))
    gtk_widget_set_parent (GTK_WIDGET (self->popover), GTK_WIDGET (self->marker_image));

  gtk_popover_popup (GTK_POPOVER (self->popover));
}

void
geoclue_stumbler_submission_marker_set_wifi_aps (GeoclueStumblerSubmissionMarker *self,
                                                  double wifi_aps)
{
  g_autofree char *label = NULL;
  label = g_strdup_printf(_("Number of WiFi APs Seen: %d"), (int)wifi_aps);
  gtk_label_set_label (self->marker_wifi_ap_label, label);
}

void
geoclue_stumbler_submission_marker_set_cell_towers (GeoclueStumblerSubmissionMarker *self,
                                                    double cell_towers)
{
  g_autofree char *label = NULL;
  label = g_strdup_printf(_("Number of Cell Towers Seen: %d"), (int)cell_towers);
  gtk_label_set_label (self->marker_cell_tower_label, label);
}

void
geoclue_stumbler_submission_marker_set_time_str (GeoclueStumblerSubmissionMarker *self,
                                                 const char *time_str)
{
  gtk_label_set_label (self->marker_time_label, time_str);
}

static void
geoclue_stumbler_submission_marker_dispose (GObject *object)
{
  GeoclueStumblerSubmissionMarker *self = (GeoclueStumblerSubmissionMarker *)object;

  if (gtk_widget_get_parent (GTK_WIDGET (self->popover)))
    gtk_widget_unparent (GTK_WIDGET (self->popover));

  G_OBJECT_CLASS (geoclue_stumbler_submission_marker_parent_class)->dispose (object);
}

static void
geoclue_stumbler_submission_marker_class_init (GeoclueStumblerSubmissionMarkerClass *klass)
{
  GObjectClass   *object_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

  gtk_widget_class_set_template_from_resource (widget_class,
                                               "/org/kop316/stumbler/geoclue-stumbler-submission-marker.ui");

  gtk_widget_class_bind_template_child (widget_class, GeoclueStumblerSubmissionMarker, popover);
  gtk_widget_class_bind_template_child (widget_class, GeoclueStumblerSubmissionMarker, marker_image);
  gtk_widget_class_bind_template_child (widget_class, GeoclueStumblerSubmissionMarker, marker_time_label);
  gtk_widget_class_bind_template_child (widget_class, GeoclueStumblerSubmissionMarker, marker_cell_tower_label);
  gtk_widget_class_bind_template_child (widget_class, GeoclueStumblerSubmissionMarker, marker_wifi_ap_label);

  object_class->dispose = geoclue_stumbler_submission_marker_dispose;
}

static void
geoclue_stumbler_submission_marker_init (GeoclueStumblerSubmissionMarker *self)
{
  GtkGesture *click_gesture = gtk_gesture_click_new ();

  gtk_widget_init_template (GTK_WIDGET (self));
  /*
   * gtk_widget_add_controller () transfers ownership of the gesture to
   * GeoclueStumblerSubmissionMarker so you will not have to worry about freeing it manually
   */
  gtk_widget_add_controller (GTK_WIDGET (self), GTK_EVENT_CONTROLLER (click_gesture));
  gtk_gesture_single_set_button (GTK_GESTURE_SINGLE (click_gesture), GDK_BUTTON_PRIMARY);
  g_signal_connect (click_gesture, "pressed", G_CALLBACK (marker_clicked_cb), self);
}

GeoclueStumblerSubmissionMarker *
geoclue_stumbler_submission_marker_new (void)
{
  return g_object_new (GEOCLUE_STUMBLER_TYPE_SUBMISSION_MARKER, NULL);
}
