/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */

/* Needed for snprintf */
#define _XOPEN_SOURCE	500

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/objects.h>
#include <openssl/err.h>

#include "_verify_proxy_certinfo.h"
#include "_verify_log.h"
#include "verify_x509_datatypes.h"


/*****************************************************************************/
/* Global static variables                                                   */
/*****************************************************************************/

/** GT3 proxy extension, the item reference can be set using
 * set_gt3_pci_type_gt3() and set_gt3_pci_type_rfc() */
static X509V3_EXT_METHOD gt3_pcert;

/*****************************************************************************/
/* Proxy Policy and certinfo prototypes                                      */
/*****************************************************************************/

/* Define PROXYPOLICY ASN1 representation, make sure this is consistent with
 * OpenSSL in crypto/x509v3/v3_pcia.c for PROXY_POLICY */
ASN1_SEQUENCE(PROXYPOLICY) = {
    ASN1_SIMPLE(PROXYPOLICY, policy_language, ASN1_OBJECT),
    ASN1_OPT(PROXYPOLICY, policy, ASN1_OCTET_STRING)
} ASN1_SEQUENCE_END(PROXYPOLICY)
IMPLEMENT_ASN1_FUNCTIONS(PROXYPOLICY)

/* Define PROXYCERTINFO ASN1 representation, make sure this is consistent with
 * OpenSSL in crypto/x509v3/v3_pcia.c for PROXY_CERT_INFO_EXTENSION */
ASN1_SEQUENCE(PROXYCERTINFO) = {
    ASN1_OPT(PROXYCERTINFO, path_length, ASN1_INTEGER),
    ASN1_SIMPLE(PROXYCERTINFO, policy, PROXYPOLICY)
} ASN1_SEQUENCE_END(PROXYCERTINFO)
IMPLEMENT_ASN1_FUNCTIONS(PROXYCERTINFO)

/* Define PROXYCERTINFO_GT3 ASN1 representation, make sure this is consistent with
 * OpenSSL in crypto/x509v3/v3_pcia.c for PROXY_CERT_INFO_EXTENSION */
ASN1_SEQUENCE(PROXYCERTINFO_GT3) = {
    ASN1_SIMPLE(PROXYCERTINFO_GT3, policy, PROXYPOLICY),
    ASN1_EXP_OPT(PROXYCERTINFO_GT3, path_length, ASN1_INTEGER, 1)
} ASN1_SEQUENCE_END(PROXYCERTINFO_GT3)
IMPLEMENT_ASN1_FUNCTIONS(PROXYCERTINFO_GT3)


/*****************************************************************************/
/* Function definitions                                                      */
/*****************************************************************************/

/* Adds a GT3 old-style proxy extension to the list of known extensions */
int init_GT3_proxy_extension(void)  {
    /* Proxy Certificate Extension's related objects */
    ERR_clear_error();
    OBJ_create(GLOBUS_PROXY_V3_OID, GLOBUS_PROXY_V3_SN, GLOBUS_PROXY_V3_LN);

    memset(&gt3_pcert, 0, sizeof(gt3_pcert));
    /* Set the GT3 NID */
    gt3_pcert.ext_nid = OBJ_txt2nid(GLOBUS_PROXY_V3_OID);
    /* If this is set the old-style ASN1 calls are ignored */
    gt3_pcert.it       = ASN1_ITEM_ref(PROXYCERTINFO_GT3);

#if 0
    gt3_pcert.ext_flags = 0;
    /* Old style ASN1 calls */
    gt3_pcert.ext_new  = (X509V3_EXT_NEW) NULL;
    gt3_pcert.ext_free = (X509V3_EXT_FREE)NULL;
    gt3_pcert.d2i      = (X509V3_EXT_D2I) NULL;  /* DER to internal */
    gt3_pcert.i2d      = (X509V3_EXT_I2D) NULL;  /* internal to DER */
    /* The following pair is used for string extensions */
    gt3_pcert.i2s      = (X509V3_EXT_I2S) NULL;  /* internal to string */
    gt3_pcert.s2i      = (X509V3_EXT_S2I) NULL;  /* string to internal */
    /* The following pair is used for multi-valued extensions */
    gt3_pcert.v2i      = (X509V3_EXT_V2I) NULL; /* STACK_OF(CONF_VALUE) to internal */
    gt3_pcert.i2v      = (X509V3_EXT_I2V) NULL; /* internal to STACK_OF(CONF_VALUE) */
    /* The following are used for raw extensions */
    gt3_pcert.i2r      = (X509V3_EXT_I2R) NULL; /* human readable to internal */
    gt3_pcert.r2i      = (X509V3_EXT_R2I) NULL; /* internal to human readable */
    /* Any extension specific data */
    gt3_pcert.usr_data = NULL;
#endif

    X509V3_EXT_add(&gt3_pcert);

    return 0;
}

/**
 * Sets the item reference for the proxy certificate info to RFC-type. caNL-java
 * produces GT3 proxies where the proxycertinfo blob is of RFC type instead of
 * the inverse ordering.
 */
void set_gt3_pci_type_rfc(void)	{
    gt3_pcert.it = ASN1_ITEM_ref(PROXYCERTINFO);
}

/**
 * Sets the item reference for the proxy certificate info to GT3-type. caNL-java
 * produces GT3 proxies where the proxycertinfo blob is of RFC type instead of
 * the inverse ordering.
 */
void set_gt3_pci_type_gt3(void)	{
    gt3_pcert.it = ASN1_ITEM_ref(PROXYCERTINFO_GT3);
}


/* Adds an RFC proxy extension to the list of known extensions */
int init_RFC_proxy_extension(void)  {
    static X509V3_EXT_METHOD pcert;

    ERR_clear_error();
    OBJ_create(RFC_PROXY_OID, RFC_PROXY_SN, RFC_PROXY_LN);

    memset(&pcert, 0, sizeof(pcert));
    /* Set the RFC NID */
    pcert.ext_nid = OBJ_txt2nid(RFC_PROXY_OID);
    /* If this is set the old-style ASN1 calls are ignored */
    pcert.it       = ASN1_ITEM_ref(PROXYCERTINFO);

#if 0
    pcert.ext_flags = 0;
    /* Old style ASN1 calls */
    pcert.ext_new  = (X509V3_EXT_NEW) NULL;
    pcert.ext_free = (X509V3_EXT_FREE)NULL;
    pcert.d2i      = (X509V3_EXT_D2I) NULL;  /* DER to internal */
    pcert.i2d      = (X509V3_EXT_I2D) NULL;  /* internal to DER */
    /* The following pair is used for string extensions */
    pcert.i2s      = (X509V3_EXT_I2S) NULL;  /* internal to string */
    pcert.s2i      = (X509V3_EXT_S2I) NULL;  /* string to internal */
    /* The following pair is used for multi-valued extensions */
    pcert.v2i      = (X509V3_EXT_V2I) NULL; /* STACK_OF(CONF_VALUE) to internal */
    pcert.i2v      = (X509V3_EXT_I2V) NULL; /* internal to STACK_OF(CONF_VALUE) */
    /* The following are used for raw extensions */
    pcert.i2r      = (X509V3_EXT_I2R) NULL; /* human readable to internal */
    pcert.r2i      = (X509V3_EXT_R2I) NULL; /* internal to human readable */
    /* Any extension specific data */
    pcert.usr_data = NULL;
#endif

    X509V3_EXT_add(&pcert);

    return 0;
}

