/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2007-2020 PCOpt/NTUA
    Copyright (C) 2013-2020 FOSS GP
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "volBSplinesBase.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

defineTypeNameAndDebug(volBSplinesBase, 0);

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

volBSplinesBase::volBSplinesBase
(
    const fvMesh& mesh
)
:
    MeshObject<fvMesh, UpdateableMeshObject, volBSplinesBase>(mesh),
    volume_(0),
    activeDesignVariables_(0)
{
    const dictionary NURBSdict
    (
        IOdictionary
        (
            IOobject
            (
                "dynamicMeshDict",
                mesh.time().constant(),
                mesh,
                IOobject::MUST_READ_IF_MODIFIED,
                IOobject::NO_WRITE,
                false
            )
        ).subDict("volumetricBSplinesMotionSolverCoeffs")
    );
    // Read box names and allocate size
    wordList controlBoxes(NURBSdict.toc());
    volume_.setSize(controlBoxes.size());

    // Populate NURBS volumes
    label iBox(0);
    for (const word& boxName : controlBoxes)
    {
        if (NURBSdict.isDict(boxName))
        {
            volume_.set
            (
                iBox,
                NURBS3DVolume::New
                (
                    NURBSdict.subDict(boxName),
                    mesh,
                    true
                )
            );
            volume_[iBox].write();
            iBox++;
        }
    }
    volume_.setSize(iBox);

    // Determine active design variables
    activeDesignVariables_.setSize(3*getTotalControlPointsNumber(), -1);
    label iActive(0);
    const labelList startCpID(getStartCpID());
    forAll(volume_, boxI)
    {
        const label start(3*startCpID[boxI]);
        const boolList& isActiveVar = volume_[boxI].getActiveDesignVariables();
        forAll(isActiveVar, varI)
        {
            if (isActiveVar[varI])
            {
                activeDesignVariables_[iActive++] = start + varI;
            }
        }
    }
    activeDesignVariables_.setSize(iActive);
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

const PtrList<NURBS3DVolume>& volBSplinesBase::boxes() const
{
    return volume_;
}


PtrList<NURBS3DVolume>& volBSplinesBase::boxesRef()
{
    return volume_;
}


const NURBS3DVolume& volBSplinesBase::box(const label boxI) const
{
    return volume_[boxI];
}


NURBS3DVolume& volBSplinesBase::boxRef(const label boxI)
{
    return volume_[boxI];
}


const vectorField& volBSplinesBase::getControlPoints(const label& iNURB) const
{
    return volume_[iNURB].getControlPoints();
}


vectorField volBSplinesBase::getAllControlPoints() const
{
    vectorField totalCPs(0);
    forAll(volume_, iNURB)
    {
        totalCPs.append(volume_[iNURB].getControlPoints());
    }

    return totalCPs;
}


Foam::label Foam::volBSplinesBase::getTotalControlPointsNumber() const
{
    label nCPs(0);
    forAll(volume_, iNURB)
    {
        nCPs += volume_[iNURB].getControlPoints().size();
    }

    return nCPs;
}


label Foam::volBSplinesBase::getNumberOfBoxes() const
{
    return volume_.size();
}


labelList volBSplinesBase::getStartCpID() const
{
    // Allocate an extra entry to track in which box a CP might be
    labelList startID(getNumberOfBoxes() + 1);
    startID[0] = 0;
    forAll(volume_, iNURB)
    {
        startID[iNURB+1] =
            startID[iNURB] + volume_[iNURB].getControlPoints().size();
    }

    return startID;
}


const Foam::labelList& volBSplinesBase::getActiveDesignVariables() const
{
    return activeDesignVariables_;
}


Foam::scalar Foam::volBSplinesBase::computeMaxBoundaryDisplacement
(
    const vectorField& controlPointsMovement,
    const labelList& patchesToBeMoved
)
{
    scalar maxDisplacement(0);
    label pastControlPoints(0);
    forAll(volume_, iNURB)
    {
        const label nb(volume_[iNURB].getControlPoints().size());
        vectorField localControlPointsMovement(nb, Zero);

        // Set localControlPointsMovement
        forAll(localControlPointsMovement, iCPM)
        {
            localControlPointsMovement[iCPM] =
                controlPointsMovement[pastControlPoints + iCPM];
        }

        maxDisplacement = max
        (
            maxDisplacement,
            volume_[iNURB].computeMaxBoundaryDisplacement
            (
                localControlPointsMovement,
                patchesToBeMoved
            )
        );

        pastControlPoints += nb;
    }

    return maxDisplacement;
}


void Foam::volBSplinesBase::boundControlPointMovement
(
    vectorField& controlPointsMovement
)
{
    label pastControlPoints(0);
    forAll(volume_, iNURB)
    {
        const label nb(volume_[iNURB].getControlPoints().size());
        vectorField localControlPointsMovement(nb, Zero);

        // Set localControlPointsMovement
        forAll (localControlPointsMovement, iCPM)
        {
            localControlPointsMovement[iCPM] =
                controlPointsMovement[pastControlPoints + iCPM];
        }

        volume_[iNURB].boundControlPointMovement(localControlPointsMovement);

        // Transfer bounding back to controlPointMovement
        forAll(localControlPointsMovement, iCPM)
        {
            controlPointsMovement[pastControlPoints + iCPM] =
                localControlPointsMovement[iCPM];
        }

        pastControlPoints += nb;
    }
}


void Foam::volBSplinesBase::moveControlPoints
(
    const vectorField& controlPointsMovement
)
{
    label pastControlPoints(0);
    forAll(volume_, iNURB)
    {
        const label nb(volume_[iNURB].getControlPoints().size());
        vectorField localControlPointsMovement(nb, Zero);

        // Set localControlPointsMovement
        forAll (localControlPointsMovement, iCPM)
        {
            localControlPointsMovement[iCPM] =
                controlPointsMovement[pastControlPoints + iCPM];
        }

        const vectorField newCps
        (
            volume_[iNURB].getControlPoints()
          + localControlPointsMovement
        );

        volume_[iNURB].setControlPoints(newCps);

        pastControlPoints += nb;
    }
}


void Foam::volBSplinesBase::writeControlPoints() const
{
    for (const NURBS3DVolume& box : volume_)
    {
        box.writeCps("cpsBsplines"+mesh_.time().timeName());
        box.writeCpsInDict();
    }
}


bool volBSplinesBase::movePoints()
{
    // Does nothing
    return true;
}


void volBSplinesBase::updateMesh(const mapPolyMesh&)
{
    // Does nothing
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// ************************************************************************* //
