/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a free video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * EBML parser
 * a lot of ideas from the gstreamer parser
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#define LOG_MODULE "ebml"
#define LOG_VERBOSE
/*
#define LOG
*/
#include <xine/xine_internal.h>
#include <xine/xineutils.h>
#include "bswap.h"

#include "ebml.h"


ebml_parser_t *new_ebml_parser (xine_t *xine, input_plugin_t *input) {
  ebml_parser_t *ebml;

  ebml = calloc(1, sizeof(ebml_parser_t));
  if (ebml) {
    ebml->xine                 = xine;
    ebml->input                = input;
  }
  return ebml;
}


void dispose_ebml_parser(ebml_parser_t *ebml) {
  if (ebml) {
    _x_freep(&ebml->doctype);
    free(ebml);
  }
}


typedef union {
  uint8_t b[8];
  uint32_t u32[2];
  uint64_t u64;
  float f;
  double d;
} ebml_value_t;

static uint32_t ebml_be_32 (uint32_t v) {
  const union { uint32_t v; uint8_t little; } is_endian = { 1 };
  return is_endian.little
    ? ((v >> 24) | ((v & 0x00ff0000) >> 8) | ((v & 0x0000ff00) << 8) | (v << 24))
    : v;
}

static uint64_t ebml_be_64 (uint64_t v) {
  const union { uint32_t v; uint8_t little; } is_endian = { 1 };
  return is_endian.little
    ? ((v >> 56) | ((v & ((uint64_t)0xff << 48)) >> 40) | ((v & ((uint64_t)0xff << 40)) >> 24) |
      ((v & ((uint64_t)0xff << 32)) >> 8) | ((v & ((uint64_t)0xff << 24)) << 8) |
      ((v & ((uint64_t)0xff << 16)) << 24) | ((v & ((uint64_t)0xff << 8)) << 40) | (v << 56))
    : v;
}

static const uint8_t ebml_sizes[256] = {
  8,7,6,6,5,5,5,5,4,4,4,4,4,4,4,4,
  3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
  2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
  2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};

static const uint8_t ebml_shift[4] = { 24,16, 8, 0 };

static const uint64_t ebml_full[8] = {
  0x7f, 0x3fff, 0x1fffff, 0x0fffffff,
  (~(uint64_t)0) >> 29, (~(uint64_t)0) >> 22, (~(uint64_t)0) >> 15, (~(uint64_t)0) >> 8
};

int ebml_read_elem_head (ebml_parser_t *ebml, ebml_elem_t *elem) {
  ebml_value_t data;
  int64_t eend;
  uint32_t size;
  int r, loglevel;

  elem->start = ebml->input->get_current_pos (ebml->input);

  r = ebml->input->read (ebml->input, data.b, 2);
  if (r != 2) {
    if (r > 0)
      elem->start += r;
    xprintf (ebml->xine, XINE_VERBOSITY_LOG, "ebml: %s at position %" PRId64 "\n",
      r ? "read error" : "EOF", elem->start);
    return 0;
  }
  size = ebml_sizes[data.b[0]];
  /* read the rest of the id */
  if (size) {
    if (size > 3) {
      xprintf (ebml->xine, XINE_VERBOSITY_LOG,
        "ebml: invalid EBML ID size (0x%x) at position %" PRId64 "\n", data.b[0], elem->start);
      return 0;
    }
    r = ebml->input->read (ebml->input, data.b + 2, size);
    if (r != (int)size) {
      if (r > 0)
        elem->start += r;
      xprintf (ebml->xine, XINE_VERBOSITY_LOG, "ebml: %s at position %" PRId64 "\n",
        r ? "read error" : "EOF", elem->start);
      return 0;
    }
  }
  /* NOTE: id includes the size bits. */
  elem->id = ebml_be_32 (data.u32[0]) >> ebml_shift[size];

  elem->start += size + 1;
  data.b[0] = data.b[size + 1];
  size = ebml_sizes[data.b[0]];
  /* read the rest of the len */
  if (size) {
    if (size > 7) {
      xprintf (ebml->xine, XINE_VERBOSITY_LOG,
        "ebml: Invalid EBML length size (0x%x) at position %" PRId64 "\n", data.b[0], elem->start);
      return 0;
    }
    r = ebml->input->read (ebml->input, data.b + 1, size);
    if (r != (int)size) {
      if (r > 0)
        elem->start += r;
      xprintf (ebml->xine, XINE_VERBOSITY_LOG, "ebml: %s at position %" PRId64 "\n",
        r ? "read error" : "EOF", elem->start);
      return 0;
    }
  }
  elem->start += size + 1;
  /* NOTE: len == ebml_full[n] is a legal way to indicate "to end of parent element". */
  loglevel = XINE_VERBOSITY_LOG;
  if (size < 4) {
    uint32_t value = (ebml_be_32 (data.u32[0]) >> ebml_shift[size]) & ebml_full[size];
    elem->len = value;
    if (value == ebml_full[size])
      elem->len = ebml_full[7], loglevel = XINE_VERBOSITY_DEBUG;
  } else {
    uint64_t value = (ebml_be_64 (data.u64) >> ebml_shift[size - 4]) & ebml_full[size];
    elem->len = value;
    if (value == ebml_full[size])
      elem->len = ebml_full[7], loglevel = XINE_VERBOSITY_DEBUG;
  }

  /* overrun paranoia */
  if (elem->start > (int64_t)ebml_full[7] - (int64_t)elem->len)
    elem->len = (int64_t)ebml_full[7] - elem->start;
  eend = elem->start + elem->len;
  for (ebml->next_level = ebml->level; ebml->next_level > 0; ebml->next_level--) {
    int64_t pend = ebml->elem_stack[ebml->next_level - 1].start + ebml->elem_stack[ebml->next_level - 1].len;

    if (pend > eend)
      break;
    if (pend < eend) {
      xprintf (ebml->xine, loglevel,
        "ebml: level %d element at pos %" PRId64 " exceeds container by %" PRId64 " bytes, truncating.\n",
        ebml->level, elem->start, eend - pend);
      eend = pend;
      pend -= elem->start;
      if (pend >= 0) {
        elem->len = pend;
      } else {
        /* i dear. head too large. */
        ebml->input->seek (ebml->input, pend, SEEK_CUR);
        elem->len = 0;
      }
    }
  }
  return 1;
}


static int ebml_read_elem_data(ebml_parser_t *ebml, void *buf, int64_t len) {
  int64_t pos;

  if (ebml->input->read(ebml->input, buf, len) == len)
    return 1;

  pos = ebml->input->get_current_pos (ebml->input);
  xprintf (ebml->xine, XINE_VERBOSITY_LOG,
    "ebml: read error at position %" PRId64 "\n", pos);
  return 0;
}


int ebml_skip(ebml_parser_t *ebml, ebml_elem_t *elem) {
  if (ebml->input->seek (ebml->input, elem->len, SEEK_CUR) >= 0)
    return 1;

  xprintf (ebml->xine, XINE_VERBOSITY_LOG,
    "ebml: seek error (failed skipping %" PRIu64 " bytes)\n", elem->len);
  return 0;
}


int ebml_read_uint(ebml_parser_t *ebml, ebml_elem_t *elem, uint64_t *num) {
  ebml_value_t data;
  uint32_t size;

  if ((elem->len < 1) || (elem->len > 8)) {
    xprintf (ebml->xine, XINE_VERBOSITY_LOG,
        "ebml: Invalid integer element size %" PRIu64 "\n", elem->len);
    return 0;
  }

  size = elem->len;
  if (!ebml_read_elem_data (ebml, data.b, size))
    return 0;
  if (size < 5) {
    *num = ebml_be_32 (data.u32[0]) >> ebml_shift[size - 1];
  } else {
    *num = ebml_be_64 (data.u64) >> ebml_shift[size - 5];
  }
  return 1;
}

#if 0
int ebml_read_sint (ebml_parser_t *ebml, ebml_elem_t  *elem, int64_t *num) {
  ebml_value_t data;
  uint32_t size;

  if ((elem->len < 1) || (elem->len > 8)) {
    xprintf (ebml->xine, XINE_VERBOSITY_LOG,
        "ebml: Invalid integer element size %" PRIu64 "\n", elem->len);
    return 0;
  }

  size = elem->len;
  if (!ebml_read_elem_data (ebml, data.b, size))
    return 0;
  if (size < 5) {
    *num = (int32_t)ebml_be_32 (data.u32[0]) >> ebml_shift[size - 1];
  } else {
    *num = (int64_t)ebml_be_64 (data.u64) >> ebml_shift[size - 5];
  }
  return 1;
}
#endif


int ebml_read_float (ebml_parser_t *ebml, ebml_elem_t *elem, double *num) {
  ebml_value_t data[2];
  uint64_t size = elem->len;

  if ((size != 4) && (size != 8) && (size != 10)) {
    xprintf (ebml->xine, XINE_VERBOSITY_LOG,
        "ebml: Invalid float element size %" PRIu64 "\n", size);
    return 0;
  }

  if (!ebml_read_elem_data (ebml, data[0].b, size))
    return 0;

  if (size == 10) {
    xprintf (ebml->xine, XINE_VERBOSITY_LOG,
        "ebml: FIXME! 10-byte floats unimplemented\n");
    return 0;
  }

  if (size == 4) {
    data[0].u32[0] = ebml_be_32 (data[0].u32[0]);
    *num = data[0].f;
  } else {
    data[0].u64 = ebml_be_64 (data[0].u64);
    *num = data[0].d;
  }
  return 1;
}

#if 0
int ebml_read_ascii(ebml_parser_t *ebml, ebml_elem_t *elem, char *str) {
  return ebml_read_elem_data (ebml, str, elem->len);
}

int ebml_read_utf8 (ebml_parser_t *ebml, ebml_elem_t *elem, char *str) {
  return ebml_read_ascii (ebml, elem, str);
}
#endif

char *ebml_alloc_read_ascii (ebml_parser_t *ebml, ebml_elem_t *elem) {
  char *text;

  if (elem->len >= 4096)
    return NULL;
  text = malloc (elem->len + 1);
  if (!text)
    return NULL;
  if (ebml_read_elem_data (ebml, text, elem->len)) {
    text[elem->len] = '\0';
    return text;
  }
  free (text);
  return NULL;
}

#if 0
int ebml_read_date (ebml_parser_t *ebml, ebml_elem_t *elem, int64_t *date) {
  return ebml_read_sint (ebml, elem, date);
}
#endif

int ebml_read_master (ebml_parser_t *ebml, ebml_elem_t *elem) {
  ebml_elem_t *top_elem;

  if (ebml->level < 0) {
    xprintf(ebml->xine, XINE_VERBOSITY_LOG,
            "ebml: invalid current level\n");
    return 0;
  }

  top_elem = &ebml->elem_stack[ebml->level];
  top_elem->start = elem->start;
  top_elem->len = elem->len;
  top_elem->id = elem->id;

  ebml->level++;
  lprintf("id: 0x%x, len: %" PRIu64 ", level: %d\n", elem->id, elem->len, ebml->level);
  if (ebml->level >= EBML_STACK_SIZE) {
    xprintf(ebml->xine, XINE_VERBOSITY_LOG,
	    "ebml: max level exceeded\n");
    return 0;
  }
  return 1;
}

int ebml_read_binary(ebml_parser_t *ebml, ebml_elem_t *elem, void *binary) {
  return ebml_read_elem_data(ebml, binary, elem->len);
}

int ebml_check_header(ebml_parser_t *ebml) {
  uint32_t next_level;
  ebml_elem_t master;

  if (!ebml_read_elem_head(ebml, &master)) {
    xprintf(ebml->xine, XINE_VERBOSITY_LOG,
            "ebml: invalid master element\n");
    return 0;
  }

  if (master.id != EBML_ID_EBML) {
    xprintf(ebml->xine, XINE_VERBOSITY_LOG,
            "ebml: invalid master element 0x%x\n", master.id);
    return 0;
  }

  if (!ebml_read_master (ebml, &master))
    return 0;

  next_level = 1;
  while (next_level == 1) {
    ebml_elem_t elem;

    if (!ebml_read_elem_head(ebml, &elem))
      return 0;

    switch (elem.id) {
      case EBML_ID_EBMLVERSION: {
        uint64_t num;

        if (!ebml_read_uint (ebml, &elem, &num))
          return 0;
        lprintf("ebml_version: %" PRIu64 "\n", num);
        ebml->version = num;
        break;
      }

      case EBML_ID_EBMLREADVERSION: {
        uint64_t num;

        if (!ebml_read_uint (ebml, &elem, &num))
          return 0;
        lprintf("ebml_read_version: %" PRIu64 "\n", num);
        if (num != EBML_VERSION)
          return 0;
        ebml->read_version = num;
        break;
      }

      case EBML_ID_EBMLMAXIDLENGTH: {
        uint64_t num;

        if (!ebml_read_uint (ebml, &elem, &num))
          return 0;
        lprintf("ebml_max_id_length: %" PRIu64 "\n", num);
        ebml->max_id_len = num;
        break;
      }

      case EBML_ID_EBMLMAXSIZELENGTH: {
        uint64_t num;

        if (!ebml_read_uint (ebml, &elem, &num))
          return 0;
        lprintf("ebml_max_size_length: %" PRIu64 "\n", num);
        ebml->max_size_len = num;
        break;
      }

      case EBML_ID_DOCTYPE: {
        char *text = ebml_alloc_read_ascii (ebml, &elem);
        if (!text)
          return 0;

        lprintf("doctype: %s\n", text);
        if (ebml->doctype)
          free (ebml->doctype);
        ebml->doctype = text;
        break;
      }

      case EBML_ID_DOCTYPEVERSION: {
        uint64_t num;

        if (!ebml_read_uint (ebml, &elem, &num))
          return 0;
        lprintf("doctype_version: %" PRIu64 "\n", num);
        ebml->doctype_version = num;
        break;
      }

      case EBML_ID_DOCTYPEREADVERSION: {
        uint64_t num;

        if (!ebml_read_uint (ebml, &elem, &num))
          return 0;
        lprintf("doctype_read_version: %" PRIu64 "\n", num);
        ebml->doctype_read_version = num;
        break;
      }

      default:
        xprintf(ebml->xine, XINE_VERBOSITY_LOG,
                "ebml: Unknown data type 0x%x in EBML header (ignored)\n", elem.id);
        if (!ebml_skip(ebml, &elem))
          return 0;
    }
    next_level = ebml->level = ebml->next_level;
  }

  return 1;
}

